/**
 * Permission is hereby granted, free of charge, to any person obtaining a copy of this
 * software and associated documentation files (the "Software"), to deal in the Software
 * without restriction, including without limitation the rights to use, copy, modify,
 * merge, publish, distribute, sublicense, and/or sell copies of the Software, and to
 * permit persons to whom the Software is furnished to do so, subject to the following
 * conditions:
 *
 * The above copyright notice and this permission notice shall be included in all copies
 * or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED,
 * INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
 * HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF
 * CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE
 * OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

package sample.service;

import com.microsoft.aad.msal4j.ClientCredentialFactory;
import com.microsoft.aad.msal4j.ClientCredentialParameters;
import com.microsoft.aad.msal4j.ConfidentialClientApplication;
import com.microsoft.aad.msal4j.IAuthenticationResult;
import com.microsoft.aad.msal4j.IClientCertificate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;

import java.net.InetSocketAddress;
import java.net.Proxy;
import java.security.PrivateKey;
import java.security.cert.X509Certificate;
import java.util.Collections;
import java.util.concurrent.CompletableFuture;

@Service
public class MsClientService {

    private final ConfidentialClientApplication confidentialClientApplication;
    private final ClientCredentialParameters clientCredentialParameters;


    public MsClientService(
            @Value("${client.app.id}") String clientId,
            @Value("${client.app.scope}") String scope,
            @Value("${morgan.stanley.oauth2.token-uri}") String tokenUri,
            @Autowired @Qualifier("msClientPrivateKey") PrivateKey privateKey,
            @Autowired @Qualifier("msClientPublicCertificate") X509Certificate publicCertificate
    ) throws Exception {
        clientCredentialParameters = ClientCredentialParameters.builder(Collections.singleton(scope)).build();
        final IClientCertificate fromCertificate = ClientCredentialFactory.createFromCertificate(privateKey, publicCertificate);
        confidentialClientApplication = ConfidentialClientApplication.builder(clientId, fromCertificate)
                .authority(tokenUri)
                .build();
    }

    public String getAccessToken() {
        final CompletableFuture<IAuthenticationResult> resultFuture = confidentialClientApplication.acquireToken(clientCredentialParameters);
        return resultFuture.join()
                .accessToken();

    }
}
